#!/usr/bin/env python3
"""
acceptance_check.py

Evaluate minimal acceptance criteria:
  - Reproducibility: max_delta_rate <= --delta_thresh
  - Normalization:  max_rowsum_residual <= --rowsum_thresh
  - Coverage:       no missing n vs D_values.csv (unless --ignore_coverage)
Writes results/acceptance.json and prints PASS/FAIL + brief reasons.
"""
import argparse, json
from pathlib import Path

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--metrics", default="results/stability_metrics.json")
    ap.add_argument("--delta_thresh", type=float, default=0.02)
    ap.add_argument("--rowsum_thresh", type=float, default=1e-12)
    ap.add_argument("--ignore_coverage", action="store_true")
    ap.add_argument("--out", default="results/acceptance.json")
    args = ap.parse_args()

    m = json.loads(Path(args.metrics).read_text(encoding="utf-8"))
    ok_delta = (m.get("max_delta_rate", 1.0) <= args.delta_thresh)
    ok_rows  = (m.get("max_rowsum_residual", 1.0) <= args.rowsum_thresh)
    missing  = m.get("coverage_missing") or []
    ok_cov   = True if args.ignore_coverage else (len(missing) == 0)

    passed = bool(ok_delta and ok_rows and ok_cov)
    reasons = []
    if not ok_delta: reasons.append(f"Δrate={m.get('max_delta_rate'):.4f} > {args.delta_thresh}")
    if not ok_rows:  reasons.append(f"|row−1|={m.get('max_rowsum_residual'):.3e} > {args.rowsum_thresh:.0e}")
    if not ok_cov:   reasons.append(f"missing n={missing}")

    out = {
        "pass": passed,
        "delta_ok": ok_delta,
        "rowsum_ok": ok_rows,
        "coverage_ok": ok_cov,
        "delta_thresh": args.delta_thresh,
        "rowsum_thresh": args.rowsum_thresh,
        "missing": missing,
        "metrics_ref": args.metrics,
    }
    Path(args.out).parent.mkdir(parents=True, exist_ok=True)
    Path(args.out).write_text(json.dumps(out, indent=2), encoding="utf-8")

    if passed:
        print("ACCEPTANCE PASS")
    else:
        print("ACCEPTANCE FAIL — " + "; ".join(reasons))

if __name__ == "__main__":
    main()
